<?php
/**
 * Orders Management
 */

$pageTitle = 'Orders Management';
require_once __DIR__ . '/includes/auth_check.php';

$db = getDB();

// Handle actions (BEFORE any output)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $orderId = intval($_POST['order_id'] ?? 0);
    
    if ($action === 'refund' && $orderId) {
        // Get order
        $stmt = $db->prepare("SELECT buyer_id, seller_id, amount, seller_earnings FROM orders WHERE id = ?");
        $stmt->execute([$orderId]);
        $order = $stmt->fetch();
        
        if ($order) {
            // Refund buyer
            $stmt = $db->prepare("SELECT wallet_balance FROM users WHERE id = ?");
            $stmt->execute([$order['buyer_id']]);
            $buyer = $stmt->fetch();
            
            $newBuyerBalance = $buyer['wallet_balance'] + $order['amount'];
            $stmt = $db->prepare("UPDATE users SET wallet_balance = ? WHERE id = ?");
            $stmt->execute([$newBuyerBalance, $order['buyer_id']]);
            
            // Deduct from seller
            $stmt = $db->prepare("SELECT wallet_balance FROM users WHERE id = ?");
            $stmt->execute([$order['seller_id']]);
            $seller = $stmt->fetch();
            
            $newSellerBalance = $seller['wallet_balance'] - $order['seller_earnings'];
            $stmt = $db->prepare("UPDATE users SET wallet_balance = ? WHERE id = ?");
            $stmt->execute([$newSellerBalance, $order['seller_id']]);
            
            // Record transactions
            $stmt = $db->prepare("
                INSERT INTO transactions (user_id, type, amount, balance_before, balance_after, reference_id, description)
                VALUES (?, 'refund', ?, ?, ?, ?, ?)
            ");
            $stmt->execute([$order['buyer_id'], $order['amount'], $buyer['wallet_balance'], $newBuyerBalance, $orderId, "Order refund"]);
            
            $stmt = $db->prepare("
                INSERT INTO transactions (user_id, type, amount, balance_before, balance_after, reference_id, description)
                VALUES (?, 'refund', ?, ?, ?, ?, ?)
            ");
            $stmt->execute([$order['seller_id'], -$order['seller_earnings'], $seller['wallet_balance'], $newSellerBalance, $orderId, "Order refund - deduction"]);
            
            // Update order status
            $stmt = $db->prepare("UPDATE orders SET status = 'refunded' WHERE id = ?");
            $stmt->execute([$orderId]);
        }
    }
    
    header('Location: orders.php');
    exit;
}

// Now include header (after POST handling)
require_once __DIR__ . '/includes/header.php';

// Get filters
$status = $_GET['status'] ?? 'all';
$search = $_GET['search'] ?? '';

$query = "
    SELECT o.*, 
           u_buyer.name as buyer_name, u_buyer.email as buyer_email,
           u_seller.name as seller_name, u_seller.email as seller_email,
           a.title as account_title
    FROM orders o
    JOIN users u_buyer ON o.buyer_id = u_buyer.id
    JOIN users u_seller ON o.seller_id = u_seller.id
    JOIN accounts a ON o.account_id = a.id
    WHERE 1=1
";

$params = [];

if ($status !== 'all') {
    $query .= " AND o.status = ?";
    $params[] = $status;
}

if (!empty($search)) {
    $query .= " AND (o.order_number LIKE ? OR u_buyer.name LIKE ? OR u_seller.name LIKE ?)";
    $searchTerm = "%{$search}%";
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

$query .= " ORDER BY o.created_at DESC LIMIT 100";

$stmt = $db->prepare($query);
$stmt->execute($params);
$orders = $stmt->fetchAll();
?>

<div class="bg-white rounded-lg shadow mb-6">
    <div class="p-6 border-b flex justify-between items-center">
        <h2 class="text-xl font-bold">Orders</h2>
        <div class="flex space-x-2">
            <input type="text" id="search" placeholder="Search..." value="<?php echo htmlspecialchars($search); ?>" class="border rounded px-3 py-2">
            <select id="statusFilter" class="border rounded px-3 py-2">
                <option value="all" <?php echo $status === 'all' ? 'selected' : ''; ?>>All Status</option>
                <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>Pending</option>
                <option value="processing" <?php echo $status === 'processing' ? 'selected' : ''; ?>>Processing</option>
                <option value="delivered" <?php echo $status === 'delivered' ? 'selected' : ''; ?>>Delivered</option>
                <option value="refunded" <?php echo $status === 'refunded' ? 'selected' : ''; ?>>Refunded</option>
            </select>
        </div>
    </div>
    
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Order #</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Buyer</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Seller</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Account</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Amount</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Date</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                <?php foreach ($orders as $order): ?>
                <tr>
                    <td class="px-6 py-4 whitespace-nowrap font-medium"><?php echo htmlspecialchars($order['order_number']); ?></td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <div>
                            <div class="font-medium"><?php echo htmlspecialchars($order['buyer_name']); ?></div>
                            <div class="text-sm text-gray-500"><?php echo htmlspecialchars($order['buyer_email']); ?></div>
                        </div>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <div>
                            <div class="font-medium"><?php echo htmlspecialchars($order['seller_name']); ?></div>
                            <div class="text-sm text-gray-500"><?php echo htmlspecialchars($order['seller_email']); ?></div>
                        </div>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm"><?php echo htmlspecialchars($order['account_title']); ?></td>
                    <td class="px-6 py-4 whitespace-nowrap font-medium">৳<?php echo number_format($order['amount'], 2); ?></td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <?php
                        $statusColors = [
                            'pending' => 'bg-yellow-100 text-yellow-800',
                            'processing' => 'bg-blue-100 text-blue-800',
                            'delivered' => 'bg-green-100 text-green-800',
                            'refunded' => 'bg-red-100 text-red-800',
                            'cancelled' => 'bg-gray-100 text-gray-800',
                        ];
                        $color = $statusColors[$order['status']] ?? 'bg-gray-100 text-gray-800';
                        ?>
                        <span class="px-2 py-1 text-xs rounded-full <?php echo $color; ?>">
                            <?php echo ucfirst($order['status']); ?>
                        </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        <?php echo date('M d, Y H:i', strtotime($order['created_at'])); ?>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm">
                        <button onclick="viewOrder(<?php echo $order['id']; ?>)" class="text-blue-600 hover:text-blue-800 mr-3">
                            <i class="fas fa-eye"></i> View
                        </button>
                        <?php if ($order['status'] === 'delivered'): ?>
                            <button onclick="refundOrder(<?php echo $order['id']; ?>)" class="text-red-600 hover:text-red-800">
                                <i class="fas fa-undo"></i> Refund
                            </button>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Order Details Modal -->
<div id="orderModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
    <div class="bg-white rounded-lg p-6 w-full max-w-2xl max-h-[90vh] overflow-y-auto">
        <h3 class="text-xl font-bold mb-4">Order Details</h3>
        <div id="orderDetails"></div>
        <div class="mt-4 flex justify-end">
            <button onclick="closeModal()" class="px-4 py-2 border rounded">Close</button>
        </div>
    </div>
</div>

<script>
function viewOrder(id) {
    // Load order details via AJAX or show in modal
    fetch(`api/orders.php?id=${id}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const order = data.data;
                document.getElementById('orderDetails').innerHTML = `
                    <div class="space-y-4">
                        <div><strong>Order Number:</strong> ${order.order_number}</div>
                        <div><strong>Buyer:</strong> ${order.buyer_name}</div>
                        <div><strong>Seller:</strong> ${order.seller_name}</div>
                        <div><strong>Amount:</strong> ৳${parseFloat(order.amount).toFixed(2)}</div>
                        <div><strong>Status:</strong> ${order.status}</div>
                        ${order.credentials ? '<div><strong>Credentials:</strong><pre>' + JSON.stringify(order.credentials, null, 2) + '</pre></div>' : ''}
                    </div>
                `;
                document.getElementById('orderModal').classList.remove('hidden');
            }
        });
}

function refundOrder(id) {
    if (confirm('Are you sure you want to refund this order? This will refund the buyer and deduct from seller.')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = '<input type="hidden" name="action" value="refund"><input type="hidden" name="order_id" value="' + id + '">';
        document.body.appendChild(form);
        form.submit();
    }
}

function closeModal() {
    document.getElementById('orderModal').classList.add('hidden');
}

document.getElementById('search').addEventListener('keypress', function(e) {
    if (e.key === 'Enter') {
        const search = this.value;
        const status = document.getElementById('statusFilter').value;
        window.location.href = `orders.php?search=${encodeURIComponent(search)}&status=${status}`;
    }
});

document.getElementById('statusFilter').addEventListener('change', function() {
    const search = document.getElementById('search').value;
    const status = this.value;
    window.location.href = `orders.php?search=${encodeURIComponent(search)}&status=${status}`;
});
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>

