<?php
/**
 * Reviews Management
 */

$pageTitle = 'Reviews Management';
require_once __DIR__ . '/includes/auth_check.php';

$db = getDB();

// Handle delete (BEFORE any output)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'delete') {
    $reviewId = intval($_POST['review_id'] ?? 0);
    if ($reviewId) {
        $stmt = $db->prepare("DELETE FROM reviews WHERE id = ?");
        $stmt->execute([$reviewId]);
    }
    header('Location: reviews.php');
    exit;
}

// Now include header (after POST handling)
require_once __DIR__ . '/includes/header.php';

// Get all reviews
$stmt = $db->query("
    SELECT r.*, 
           u_buyer.name as buyer_name,
           u_seller.name as seller_name,
           o.order_number
    FROM reviews r
    JOIN users u_buyer ON r.buyer_id = u_buyer.id
    JOIN users u_seller ON r.seller_id = u_seller.id
    JOIN orders o ON r.order_id = o.id
    ORDER BY r.created_at DESC
");
$reviews = $stmt->fetchAll();
?>

<div class="bg-white rounded-lg shadow">
    <div class="p-6 border-b">
        <h2 class="text-xl font-bold">Reviews</h2>
    </div>
    
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Order</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Buyer</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Seller</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Rating</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Comment</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Date</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                <?php foreach ($reviews as $review): ?>
                <tr>
                    <td class="px-6 py-4 whitespace-nowrap text-sm"><?php echo htmlspecialchars($review['order_number']); ?></td>
                    <td class="px-6 py-4 whitespace-nowrap"><?php echo htmlspecialchars($review['buyer_name']); ?></td>
                    <td class="px-6 py-4 whitespace-nowrap"><?php echo htmlspecialchars($review['seller_name']); ?></td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <div class="flex items-center">
                            <?php for ($i = 1; $i <= 5; $i++): ?>
                                <i class="fas fa-star <?php echo $i <= $review['rating'] ? 'text-yellow-400' : 'text-gray-300'; ?>"></i>
                            <?php endfor; ?>
                            <span class="ml-2"><?php echo $review['rating']; ?>/5</span>
                        </div>
                    </td>
                    <td class="px-6 py-4">
                        <div class="max-w-md"><?php echo htmlspecialchars($review['comment'] ?? 'No comment'); ?></div>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        <?php echo date('M d, Y', strtotime($review['created_at'])); ?>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm">
                        <button onclick="deleteReview(<?php echo $review['id']; ?>)" class="text-red-600 hover:text-red-800">
                            <i class="fas fa-trash"></i> Delete
                        </button>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<script>
function deleteReview(id) {
    if (confirm('Are you sure you want to delete this review?')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = '<input type="hidden" name="action" value="delete"><input type="hidden" name="review_id" value="' + id + '">';
        document.body.appendChild(form);
        form.submit();
    }
}
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>

