<?php
/**
 * Social Media Types Management
 */

$pageTitle = 'Social Media Types';
require_once __DIR__ . '/includes/auth_check.php';

$db = getDB();

// Handle actions (BEFORE any output)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    if ($action === 'add') {
        $name = trim($_POST['name'] ?? '');
        $slug = strtolower(trim($_POST['slug'] ?? ''));
        $commission = floatval($_POST['commission_percent'] ?? 5.00);
        $minPrice = floatval($_POST['min_price'] ?? 0);
        $maxPrice = floatval($_POST['max_price'] ?? 10000);
        $isActive = isset($_POST['is_active']) ? 1 : 0;
        
        if (!empty($name) && !empty($slug)) {
            $stmt = $db->prepare("
                INSERT INTO social_media_types (name, slug, commission_percent, min_price, max_price, is_active)
                VALUES (?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([$name, $slug, $commission, $minPrice, $maxPrice, $isActive]);
        }
    } elseif ($action === 'edit' && isset($_POST['id'])) {
        $id = intval($_POST['id']);
        $name = trim($_POST['name'] ?? '');
        $slug = strtolower(trim($_POST['slug'] ?? ''));
        $commission = floatval($_POST['commission_percent'] ?? 5.00);
        $minPrice = floatval($_POST['min_price'] ?? 0);
        $maxPrice = floatval($_POST['max_price'] ?? 10000);
        $isActive = isset($_POST['is_active']) ? 1 : 0;
        
        $stmt = $db->prepare("
            UPDATE social_media_types 
            SET name = ?, slug = ?, commission_percent = ?, min_price = ?, max_price = ?, is_active = ?
            WHERE id = ?
        ");
        $stmt->execute([$name, $slug, $commission, $minPrice, $maxPrice, $isActive, $id]);
    } elseif ($action === 'delete' && isset($_POST['id'])) {
        $id = intval($_POST['id']);
        $stmt = $db->prepare("DELETE FROM social_media_types WHERE id = ?");
        $stmt->execute([$id]);
    }
    
    header('Location: social_types.php');
    exit;
}

// Now include header (after POST handling)
require_once __DIR__ . '/includes/header.php';

// Get all social types
$stmt = $db->query("
    SELECT st.*, 
           (SELECT COUNT(*) FROM accounts a WHERE a.social_type_id = st.id AND a.is_active = 1) as active_accounts,
           (SELECT COUNT(*) FROM social_media_fields f WHERE f.social_type_id = st.id) as field_count
    FROM social_media_types st
    ORDER BY st.created_at DESC
");
$socialTypes = $stmt->fetchAll();
?>

<div class="bg-white rounded-lg shadow mb-6">
    <div class="p-6 border-b flex justify-between items-center">
        <h2 class="text-xl font-bold">Social Media Types</h2>
        <button onclick="showAddModal()" class="bg-blue-500 hover:bg-blue-700 text-white font-bold py-2 px-4 rounded">
            <i class="fas fa-plus"></i> Add New
        </button>
    </div>
    
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Name</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Slug</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Commission</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Price Range</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Accounts</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Fields</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                <?php foreach ($socialTypes as $type): ?>
                <tr>
                    <td class="px-6 py-4 whitespace-nowrap font-medium"><?php echo htmlspecialchars($type['name']); ?></td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500"><?php echo htmlspecialchars($type['slug']); ?></td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm"><?php echo number_format($type['commission_percent'], 2); ?>%</td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm">৳<?php echo number_format($type['min_price'], 2); ?> - ৳<?php echo number_format($type['max_price'], 2); ?></td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm"><?php echo $type['active_accounts']; ?></td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm"><?php echo $type['field_count']; ?></td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <?php if ($type['is_active']): ?>
                            <span class="px-2 py-1 text-xs rounded-full bg-green-100 text-green-800">Active</span>
                        <?php else: ?>
                            <span class="px-2 py-1 text-xs rounded-full bg-red-100 text-red-800">Inactive</span>
                        <?php endif; ?>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm">
                        <button onclick="editType(<?php echo htmlspecialchars(json_encode($type)); ?>)" class="text-blue-600 hover:text-blue-800 mr-3">
                            <i class="fas fa-edit"></i> Edit
                        </button>
                        <button onclick="deleteType(<?php echo $type['id']; ?>)" class="text-red-600 hover:text-red-800">
                            <i class="fas fa-trash"></i> Delete
                        </button>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Add/Edit Modal -->
<div id="typeModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
    <div class="bg-white rounded-lg p-6 w-full max-w-md">
        <h3 class="text-xl font-bold mb-4" id="modalTitle">Add Social Media Type</h3>
        <form method="POST" id="typeForm">
            <input type="hidden" name="action" id="formAction" value="add">
            <input type="hidden" name="id" id="formId">
            
            <div class="mb-4">
                <label class="block text-sm font-medium mb-2">Name</label>
                <input type="text" name="name" id="formName" required class="w-full border rounded px-3 py-2">
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium mb-2">Slug</label>
                <input type="text" name="slug" id="formSlug" required class="w-full border rounded px-3 py-2">
            </div>
            
            <div class="mb-4">
                <label class="block text-sm font-medium mb-2">Commission %</label>
                <input type="number" step="0.01" name="commission_percent" id="formCommission" value="5.00" required class="w-full border rounded px-3 py-2">
            </div>
            
            <div class="grid grid-cols-2 gap-4 mb-4">
                <div>
                    <label class="block text-sm font-medium mb-2">Min Price</label>
                    <input type="number" step="0.01" name="min_price" id="formMinPrice" value="0" required class="w-full border rounded px-3 py-2">
                </div>
                <div>
                    <label class="block text-sm font-medium mb-2">Max Price</label>
                    <input type="number" step="0.01" name="max_price" id="formMaxPrice" value="10000" required class="w-full border rounded px-3 py-2">
                </div>
            </div>
            
            <div class="mb-4">
                <label class="flex items-center">
                    <input type="checkbox" name="is_active" id="formIsActive" checked class="mr-2">
                    <span>Active</span>
                </label>
            </div>
            
            <div class="flex justify-end space-x-2">
                <button type="button" onclick="closeModal()" class="px-4 py-2 border rounded">Cancel</button>
                <button type="submit" class="px-4 py-2 bg-blue-500 text-white rounded">Save</button>
            </div>
        </form>
    </div>
</div>

<script>
function showAddModal() {
    document.getElementById('modalTitle').textContent = 'Add Social Media Type';
    document.getElementById('formAction').value = 'add';
    document.getElementById('typeForm').reset();
    document.getElementById('formId').value = '';
    document.getElementById('typeModal').classList.remove('hidden');
}

function editType(type) {
    document.getElementById('modalTitle').textContent = 'Edit Social Media Type';
    document.getElementById('formAction').value = 'edit';
    document.getElementById('formId').value = type.id;
    document.getElementById('formName').value = type.name;
    document.getElementById('formSlug').value = type.slug;
    document.getElementById('formCommission').value = type.commission_percent;
    document.getElementById('formMinPrice').value = type.min_price;
    document.getElementById('formMaxPrice').value = type.max_price;
    document.getElementById('formIsActive').checked = type.is_active == 1;
    document.getElementById('typeModal').classList.remove('hidden');
}

function deleteType(id) {
    if (confirm('Are you sure you want to delete this social media type? This will also delete all associated fields and accounts.')) {
        const form = document.createElement('form');
        form.method = 'POST';
        form.innerHTML = '<input type="hidden" name="action" value="delete"><input type="hidden" name="id" value="' + id + '">';
        document.body.appendChild(form);
        form.submit();
    }
}

function closeModal() {
    document.getElementById('typeModal').classList.add('hidden');
}
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>

