<?php
/**
 * Users Management
 */

$pageTitle = 'Users Management';
require_once __DIR__ . '/includes/auth_check.php';

$db = getDB();

// Handle actions (BEFORE any output)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $userId = $_POST['user_id'] ?? null;
    
    if ($action === 'block' && $userId) {
        $stmt = $db->prepare("UPDATE users SET is_blocked = 1 WHERE id = ?");
        $stmt->execute([$userId]);
    } elseif ($action === 'unblock' && $userId) {
        $stmt = $db->prepare("UPDATE users SET is_blocked = 0 WHERE id = ?");
        $stmt->execute([$userId]);
    } elseif ($action === 'adjust_balance' && $userId) {
        $amount = floatval($_POST['amount'] ?? 0);
        $reason = trim($_POST['reason'] ?? '');
        
        if ($amount != 0 && !empty($reason)) {
            $stmt = $db->prepare("SELECT wallet_balance FROM users WHERE id = ?");
            $stmt->execute([$userId]);
            $user = $stmt->fetch();
            
            $newBalance = $user['wallet_balance'] + $amount;
            $stmt = $db->prepare("UPDATE users SET wallet_balance = ? WHERE id = ?");
            $stmt->execute([$newBalance, $userId]);
            
            // Record transaction
            $stmt = $db->prepare("
                INSERT INTO transactions (user_id, type, amount, balance_before, balance_after, admin_note, description)
                VALUES (?, 'adjustment', ?, ?, ?, ?, ?)
            ");
            $stmt->execute([$userId, $amount, $user['wallet_balance'], $newBalance, $reason, "Admin balance adjustment: {$reason}"]);
        }
    }
    
    header('Location: users.php');
    exit;
}

// Now include header (after POST handling)
require_once __DIR__ . '/includes/header.php';

// Get users
$role = $_GET['role'] ?? 'all';
$search = $_GET['search'] ?? '';

$query = "SELECT id, name, email, role, wallet_balance, is_verified, is_blocked, created_at FROM users WHERE role != 'admin'";
$params = [];

if ($role !== 'all') {
    $query .= " AND role = ?";
    $params[] = $role;
}

if (!empty($search)) {
    $query .= " AND (name LIKE ? OR email LIKE ?)";
    $searchTerm = "%{$search}%";
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

$query .= " ORDER BY created_at DESC";

$stmt = $db->prepare($query);
$stmt->execute($params);
$users = $stmt->fetchAll();
?>

<div class="bg-white rounded-lg shadow mb-6">
    <div class="p-6 border-b flex justify-between items-center">
        <h2 class="text-xl font-bold">Users</h2>
        <div class="flex space-x-2">
            <input type="text" id="search" placeholder="Search..." value="<?php echo htmlspecialchars($search); ?>"
                   class="border rounded px-3 py-2">
            <select id="roleFilter" class="border rounded px-3 py-2">
                <option value="all" <?php echo $role === 'all' ? 'selected' : ''; ?>>All Roles</option>
                <option value="buyer" <?php echo $role === 'buyer' ? 'selected' : ''; ?>>Buyers</option>
                <option value="seller" <?php echo $role === 'seller' ? 'selected' : ''; ?>>Sellers</option>
            </select>
        </div>
    </div>
    
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Name</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Email</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Role</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Balance</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                <?php foreach ($users as $user): ?>
                <tr>
                    <td class="px-6 py-4 whitespace-nowrap"><?php echo htmlspecialchars($user['name']); ?></td>
                    <td class="px-6 py-4 whitespace-nowrap"><?php echo htmlspecialchars($user['email']); ?></td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <span class="px-2 py-1 text-xs rounded-full <?php echo $user['role'] === 'seller' ? 'bg-blue-100 text-blue-800' : 'bg-gray-100 text-gray-800'; ?>">
                            <?php echo ucfirst($user['role']); ?>
                        </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap">৳<?php echo number_format($user['wallet_balance'], 2); ?></td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <?php if ($user['is_blocked']): ?>
                            <span class="px-2 py-1 text-xs rounded-full bg-red-100 text-red-800">Blocked</span>
                        <?php else: ?>
                            <span class="px-2 py-1 text-xs rounded-full bg-green-100 text-green-800">Active</span>
                        <?php endif; ?>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm">
                        <div class="flex space-x-2">
                            <?php if ($user['is_blocked']): ?>
                                <form method="POST" class="inline">
                                    <input type="hidden" name="action" value="unblock">
                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                    <button type="submit" class="text-green-600 hover:text-green-800">Unblock</button>
                                </form>
                            <?php else: ?>
                                <form method="POST" class="inline">
                                    <input type="hidden" name="action" value="block">
                                    <input type="hidden" name="user_id" value="<?php echo $user['id']; ?>">
                                    <button type="submit" class="text-red-600 hover:text-red-800">Block</button>
                                </form>
                            <?php endif; ?>
                            <button onclick="adjustBalance(<?php echo $user['id']; ?>, '<?php echo htmlspecialchars($user['name']); ?>')" 
                                    class="text-blue-600 hover:text-blue-800">Adjust Balance</button>
                        </div>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Balance Adjustment Modal -->
<div id="balanceModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
    <div class="bg-white rounded-lg p-6 w-full max-w-md">
        <h3 class="text-xl font-bold mb-4">Adjust Balance</h3>
        <form method="POST">
            <input type="hidden" name="action" value="adjust_balance">
            <input type="hidden" name="user_id" id="adjust_user_id">
            <div class="mb-4">
                <label class="block text-sm font-medium mb-2">User</label>
                <input type="text" id="adjust_user_name" readonly class="w-full border rounded px-3 py-2 bg-gray-100">
            </div>
            <div class="mb-4">
                <label class="block text-sm font-medium mb-2">Amount (positive to add, negative to subtract)</label>
                <input type="number" step="0.01" name="amount" required class="w-full border rounded px-3 py-2">
            </div>
            <div class="mb-4">
                <label class="block text-sm font-medium mb-2">Reason</label>
                <textarea name="reason" required class="w-full border rounded px-3 py-2"></textarea>
            </div>
            <div class="flex justify-end space-x-2">
                <button type="button" onclick="document.getElementById('balanceModal').classList.add('hidden')" 
                        class="px-4 py-2 border rounded">Cancel</button>
                <button type="submit" class="px-4 py-2 bg-blue-500 text-white rounded">Submit</button>
            </div>
        </form>
    </div>
</div>

<script>
function adjustBalance(userId, userName) {
    document.getElementById('adjust_user_id').value = userId;
    document.getElementById('adjust_user_name').value = userName;
    document.getElementById('balanceModal').classList.remove('hidden');
}

document.getElementById('search').addEventListener('keypress', function(e) {
    if (e.key === 'Enter') {
        const search = this.value;
        const role = document.getElementById('roleFilter').value;
        window.location.href = `users.php?search=${encodeURIComponent(search)}&role=${role}`;
    }
});

document.getElementById('roleFilter').addEventListener('change', function() {
    const search = document.getElementById('search').value;
    const role = this.value;
    window.location.href = `users.php?search=${encodeURIComponent(search)}&role=${role}`;
});
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>

