<?php
/**
 * Withdrawals Management
 */

$pageTitle = 'Withdrawals Management';
require_once __DIR__ . '/includes/auth_check.php';

$db = getDB();

// Handle actions (BEFORE any output)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    $withdrawId = intval($_POST['withdraw_id'] ?? 0);
    $adminNote = trim($_POST['admin_note'] ?? '');
    
    if ($action === 'approve' && $withdrawId) {
        // Get withdrawal
        $stmt = $db->prepare("
            SELECT wr.user_id, wr.amount 
            FROM withdraw_requests wr 
            WHERE wr.id = ? AND wr.status = 'pending'
        ");
        $stmt->execute([$withdrawId]);
        $withdraw = $stmt->fetch();
        
        if ($withdraw) {
            // Update withdrawal status
            $stmt = $db->prepare("
                UPDATE withdraw_requests 
                SET status = 'approved', admin_note = ?, processed_at = NOW() 
                WHERE id = ?
            ");
            $stmt->execute([$adminNote, $withdrawId]);
            
            // Record transaction (balance was already deducted when request was created)
            $stmt = $db->prepare("
                INSERT INTO transactions (user_id, type, amount, reference_id, description, admin_note)
                VALUES (?, 'withdraw', ?, ?, ?, ?)
            ");
            $stmt->execute([
                $withdraw['user_id'],
                -$withdraw['amount'],
                $withdrawId,
                "Withdrawal approved",
                $adminNote
            ]);
            
            // Create notification
            $stmt = $db->prepare("
                INSERT INTO notifications (user_id, title, message, type)
                VALUES (?, 'Withdrawal Approved', ?, 'success')
            ");
            $stmt->execute([$withdraw['user_id'], "Your withdrawal of ৳{$withdraw['amount']} has been processed"]);
        }
    } elseif ($action === 'reject' && $withdrawId) {
        // Get withdrawal
        $stmt = $db->prepare("SELECT user_id, amount FROM withdraw_requests WHERE id = ? AND status = 'pending'");
        $stmt->execute([$withdrawId]);
        $withdraw = $stmt->fetch();
        
        if ($withdraw) {
            // Refund to wallet
            $stmt = $db->prepare("SELECT wallet_balance FROM users WHERE id = ?");
            $stmt->execute([$withdraw['user_id']]);
            $user = $stmt->fetch();
            
            $newBalance = $user['wallet_balance'] + $withdraw['amount'];
            $stmt = $db->prepare("UPDATE users SET wallet_balance = ? WHERE id = ?");
            $stmt->execute([$newBalance, $withdraw['user_id']]);
            
            // Record transaction
            $stmt = $db->prepare("
                INSERT INTO transactions (user_id, type, amount, balance_before, balance_after, reference_id, description, admin_note)
                VALUES (?, 'refund', ?, ?, ?, ?, ?, ?)
            ");
            $stmt->execute([
                $withdraw['user_id'],
                $withdraw['amount'],
                $user['wallet_balance'],
                $newBalance,
                $withdrawId,
                "Withdrawal rejected - refunded",
                $adminNote
            ]);
            
            // Update withdrawal status
            $stmt = $db->prepare("UPDATE withdraw_requests SET status = 'rejected', admin_note = ? WHERE id = ?");
            $stmt->execute([$adminNote, $withdrawId]);
            
            // Create notification
            $stmt = $db->prepare("
                INSERT INTO notifications (user_id, title, message, type)
                VALUES (?, 'Withdrawal Rejected', ?, 'error')
            ");
            $stmt->execute([$withdraw['user_id'], "Your withdrawal of ৳{$withdraw['amount']} has been rejected. Amount refunded to wallet."]);
        }
    }
    
    header('Location: withdrawals.php');
    exit;
}

// Now include header (after POST handling)
require_once __DIR__ . '/includes/header.php';

// Get filters
$status = $_GET['status'] ?? 'all';
$search = $_GET['search'] ?? '';

$query = "
    SELECT wr.*, u.name as user_name, u.email as user_email, wg.name as gateway_name, wg.type as gateway_type
    FROM withdraw_requests wr
    JOIN users u ON wr.user_id = u.id
    JOIN withdraw_gateways wg ON wr.gateway_id = wg.id
    WHERE 1=1
";

$params = [];

if ($status !== 'all') {
    $query .= " AND wr.status = ?";
    $params[] = $status;
}

if (!empty($search)) {
    $query .= " AND (u.name LIKE ? OR u.email LIKE ? OR wr.account_number LIKE ?)";
    $searchTerm = "%{$search}%";
    $params[] = $searchTerm;
    $params[] = $searchTerm;
    $params[] = $searchTerm;
}

$query .= " ORDER BY wr.created_at DESC";

$stmt = $db->prepare($query);
$stmt->execute($params);
$withdrawals = $stmt->fetchAll();
?>

<div class="bg-white rounded-lg shadow mb-6">
    <div class="p-6 border-b flex justify-between items-center">
        <h2 class="text-xl font-bold">Withdrawals</h2>
        <div class="flex space-x-2">
            <input type="text" id="search" placeholder="Search..." value="<?php echo htmlspecialchars($search); ?>" class="border rounded px-3 py-2">
            <select id="statusFilter" class="border rounded px-3 py-2">
                <option value="all" <?php echo $status === 'all' ? 'selected' : ''; ?>>All Status</option>
                <option value="pending" <?php echo $status === 'pending' ? 'selected' : ''; ?>>Pending</option>
                <option value="approved" <?php echo $status === 'approved' ? 'selected' : ''; ?>>Approved</option>
                <option value="rejected" <?php echo $status === 'rejected' ? 'selected' : ''; ?>>Rejected</option>
            </select>
        </div>
    </div>
    
    <div class="overflow-x-auto">
        <table class="w-full">
            <thead class="bg-gray-50">
                <tr>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">User</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Amount</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Gateway</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Account Number</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Status</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Date</th>
                    <th class="px-6 py-3 text-left text-xs font-medium text-gray-500 uppercase">Actions</th>
                </tr>
            </thead>
            <tbody class="divide-y divide-gray-200">
                <?php foreach ($withdrawals as $withdraw): ?>
                <tr>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <div>
                            <div class="font-medium"><?php echo htmlspecialchars($withdraw['user_name']); ?></div>
                            <div class="text-sm text-gray-500"><?php echo htmlspecialchars($withdraw['user_email']); ?></div>
                        </div>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap font-medium">৳<?php echo number_format($withdraw['amount'], 2); ?></td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <div>
                            <div class="font-medium"><?php echo htmlspecialchars($withdraw['gateway_name']); ?></div>
                            <div class="text-sm text-gray-500"><?php echo htmlspecialchars($withdraw['gateway_type']); ?></div>
                        </div>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm"><?php echo htmlspecialchars($withdraw['account_number']); ?></td>
                    <td class="px-6 py-4 whitespace-nowrap">
                        <?php
                        $statusColors = [
                            'pending' => 'bg-yellow-100 text-yellow-800',
                            'approved' => 'bg-green-100 text-green-800',
                            'rejected' => 'bg-red-100 text-red-800',
                        ];
                        $color = $statusColors[$withdraw['status']] ?? 'bg-gray-100 text-gray-800';
                        ?>
                        <span class="px-2 py-1 text-xs rounded-full <?php echo $color; ?>">
                            <?php echo ucfirst($withdraw['status']); ?>
                        </span>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm text-gray-500">
                        <?php echo date('M d, Y H:i', strtotime($withdraw['created_at'])); ?>
                    </td>
                    <td class="px-6 py-4 whitespace-nowrap text-sm">
                        <?php if ($withdraw['status'] === 'pending'): ?>
                            <button onclick="approveWithdraw(<?php echo $withdraw['id']; ?>)" class="text-green-600 hover:text-green-800 mr-3">
                                <i class="fas fa-check"></i> Approve
                            </button>
                            <button onclick="rejectWithdraw(<?php echo $withdraw['id']; ?>)" class="text-red-600 hover:text-red-800">
                                <i class="fas fa-times"></i> Reject
                            </button>
                        <?php else: ?>
                            <?php if ($withdraw['admin_note']): ?>
                                <span class="text-gray-500" title="<?php echo htmlspecialchars($withdraw['admin_note']); ?>">
                                    <i class="fas fa-info-circle"></i>
                                </span>
                            <?php endif; ?>
                        <?php endif; ?>
                    </td>
                </tr>
                <?php endforeach; ?>
            </tbody>
        </table>
    </div>
</div>

<!-- Action Modal -->
<div id="actionModal" class="hidden fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
    <div class="bg-white rounded-lg p-6 w-full max-w-md">
        <h3 class="text-xl font-bold mb-4" id="modalTitle">Action</h3>
        <form method="POST" id="actionForm">
            <input type="hidden" name="action" id="formAction">
            <input type="hidden" name="withdraw_id" id="formWithdrawId">
            
            <div class="mb-4">
                <label class="block text-sm font-medium mb-2">Admin Note</label>
                <textarea name="admin_note" id="formNote" class="w-full border rounded px-3 py-2" rows="3"></textarea>
            </div>
            
            <div class="flex justify-end space-x-2">
                <button type="button" onclick="closeModal()" class="px-4 py-2 border rounded">Cancel</button>
                <button type="submit" class="px-4 py-2 bg-blue-500 text-white rounded" id="submitBtn">Confirm</button>
            </div>
        </form>
    </div>
</div>

<script>
function approveWithdraw(id) {
    document.getElementById('modalTitle').textContent = 'Approve Withdrawal';
    document.getElementById('formAction').value = 'approve';
    document.getElementById('formWithdrawId').value = id;
    document.getElementById('submitBtn').className = 'px-4 py-2 bg-green-500 text-white rounded';
    document.getElementById('actionModal').classList.remove('hidden');
}

function rejectWithdraw(id) {
    document.getElementById('modalTitle').textContent = 'Reject Withdrawal';
    document.getElementById('formAction').value = 'reject';
    document.getElementById('formWithdrawId').value = id;
    document.getElementById('submitBtn').className = 'px-4 py-2 bg-red-500 text-white rounded';
    document.getElementById('actionModal').classList.remove('hidden');
}

function closeModal() {
    document.getElementById('actionModal').classList.add('hidden');
}

document.getElementById('search').addEventListener('keypress', function(e) {
    if (e.key === 'Enter') {
        const search = this.value;
        const status = document.getElementById('statusFilter').value;
        window.location.href = `withdrawals.php?search=${encodeURIComponent(search)}&status=${status}`;
    }
});

document.getElementById('statusFilter').addEventListener('change', function() {
    const search = document.getElementById('search').value;
    const status = this.value;
    window.location.href = `withdrawals.php?search=${encodeURIComponent(search)}&status=${status}`;
});
</script>

<?php require_once __DIR__ . '/includes/footer.php'; ?>

