<?php
/**
 * Accounts API
 * Seller account listings management
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/middleware.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../encryption.php';

$db = getDB();
$method = $_SERVER['REQUEST_METHOD'];

// List accounts (public or seller's own)
if ($method === 'GET') {
    $sellerId = $_GET['seller_id'] ?? null;
    $socialTypeId = $_GET['social_type_id'] ?? null;
    $minPrice = $_GET['min_price'] ?? null;
    $maxPrice = $_GET['max_price'] ?? null;
    $includeInactive = isset($_GET['include_inactive']) && $_GET['include_inactive'] == '1';
    
    // Check if user is authenticated and is the seller
    $isOwnAccount = false;
    $user = null;
    try {
        $user = requireAuth();
        if ($sellerId && $user['id'] == $sellerId) {
            $isOwnAccount = true;
        }
    } catch (Exception $e) {
        // Not authenticated - public access
    }
    
    $query = "
        SELECT a.id, a.seller_id, a.social_type_id, a.title, a.description, 
               a.price, a.stock, a.sold_count, a.screenshots, a.is_active, a.created_at,
               u.name as seller_name, u.avatar as seller_avatar,
               st.name as social_type_name, st.slug as social_type_slug, st.icon,
               (SELECT AVG(rating) FROM reviews r WHERE r.seller_id = a.seller_id) as seller_rating,
               (SELECT COUNT(*) FROM reviews r WHERE r.seller_id = a.seller_id) as seller_reviews
        FROM accounts a
        JOIN users u ON a.seller_id = u.id
        JOIN social_media_types st ON a.social_type_id = st.id
        WHERE 1=1
    ";
    
    $params = [];
    
    // If not own account, only show active with stock
    if (!$isOwnAccount && !$includeInactive) {
        $query .= " AND a.is_active = 1 AND a.stock > 0";
    } elseif ($isOwnAccount && !$includeInactive) {
        // Seller viewing own accounts - show all active
        $query .= " AND a.is_active = 1";
    }
    
    if ($sellerId) {
        $query .= " AND a.seller_id = ?";
        $params[] = $sellerId;
    }
    
    if ($socialTypeId) {
        $query .= " AND a.social_type_id = ?";
        $params[] = $socialTypeId;
    }
    
    if ($minPrice !== null) {
        $query .= " AND a.price >= ?";
        $params[] = $minPrice;
    }
    
    if ($maxPrice !== null) {
        $query .= " AND a.price <= ?";
        $params[] = $maxPrice;
    }
    
    $query .= " ORDER BY a.created_at DESC";
    
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    $accounts = $stmt->fetchAll();
    
    // Parse screenshots JSON
    foreach ($accounts as &$account) {
        $account['screenshots'] = json_decode($account['screenshots'] ?? '[]', true);
        $account['seller_rating'] = (float)($account['seller_rating'] ?? 0);
        $account['seller_reviews'] = (int)($account['seller_reviews'] ?? 0);
    }
    
    echo json_encode(['success' => true, 'data' => $accounts]);
    exit;
}

// Get single account (with decrypted credentials for buyer after purchase)
if ($method === 'GET' && isset($_GET['id'])) {
    $accountId = $_GET['id'];
    $user = requireAuth();
    
    $stmt = $db->prepare("
        SELECT a.*, u.name as seller_name, st.name as social_type_name
        FROM accounts a
        JOIN users u ON a.seller_id = u.id
        JOIN social_media_types st ON a.social_type_id = st.id
        WHERE a.id = ?
    ");
    $stmt->execute([$accountId]);
    $account = $stmt->fetch();
    
    if (!$account) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Account not found']);
        exit;
    }
    
    // Check if user purchased this account
    $stmt = $db->prepare("SELECT id FROM orders WHERE account_id = ? AND buyer_id = ? AND status = 'delivered'");
    $stmt->execute([$accountId, $user['id']]);
    $order = $stmt->fetch();
    
    $account['screenshots'] = json_decode($account['screenshots'] ?? '[]', true);
    
    // Only decrypt credentials if user purchased it or is the seller
    if ($order || $account['seller_id'] == $user['id'] || $user['role'] === 'admin') {
        $account['credentials'] = Encryption::decryptJson($account['credentials_data']);
    } else {
        unset($account['credentials_data']);
    }
    
    echo json_encode(['success' => true, 'data' => $account]);
    exit;
}

// Create account (Seller only)
if ($method === 'POST') {
    $user = requireSeller();
    $input = json_decode(file_get_contents('php://input'), true);
    
    $socialTypeId = $input['social_type_id'] ?? null;
    $title = trim($input['title'] ?? '');
    $description = $input['description'] ?? '';
    $price = $input['price'] ?? null;
    $stock = $input['stock'] ?? 1;
    $credentials = $input['credentials'] ?? [];
    $screenshots = $input['screenshots'] ?? [];
    
    if (empty($socialTypeId) || empty($title) || $price === null || empty($credentials)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Missing required fields']);
        exit;
    }
    
    // Encrypt credentials
    $encryptedCredentials = Encryption::encryptJson($credentials);
    
    $stmt = $db->prepare("
        INSERT INTO accounts (seller_id, social_type_id, title, description, price, stock, credentials_data, screenshots)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?)
    ");
    
    $screenshotsJson = json_encode($screenshots);
    
    if ($stmt->execute([$user['id'], $socialTypeId, $title, $description, $price, $stock, $encryptedCredentials, $screenshotsJson])) {
        $accountId = $db->lastInsertId();
        echo json_encode(['success' => true, 'message' => 'Account created', 'data' => ['id' => $accountId]]);
    } else {
        http_response_code(500);
        echo json_encode(['success' => false, 'message' => 'Failed to create account']);
    }
    exit;
}

// Update account (Seller only)
if ($method === 'PUT' && isset($_GET['id'])) {
    $user = requireSeller();
    $accountId = $_GET['id'];
    $input = json_decode(file_get_contents('php://input'), true);
    
    // Verify ownership
    $stmt = $db->prepare("SELECT seller_id FROM accounts WHERE id = ?");
    $stmt->execute([$accountId]);
    $account = $stmt->fetch();
    
    if (!$account || $account['seller_id'] != $user['id']) {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'Not authorized']);
        exit;
    }
    
    $title = trim($input['title'] ?? '');
    $description = $input['description'] ?? '';
    $price = $input['price'] ?? null;
    $stock = $input['stock'] ?? null;
    $isActive = $input['is_active'] ?? null;
    
    $updates = [];
    $params = [];
    
    if ($title) {
        $updates[] = "title = ?";
        $params[] = $title;
    }
    if ($description !== null) {
        $updates[] = "description = ?";
        $params[] = $description;
    }
    if ($price !== null) {
        $updates[] = "price = ?";
        $params[] = $price;
    }
    if ($stock !== null) {
        $updates[] = "stock = ?";
        $params[] = $stock;
    }
    if ($isActive !== null) {
        $updates[] = "is_active = ?";
        $params[] = $isActive ? 1 : 0;
    }
    
    if (empty($updates)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'No fields to update']);
        exit;
    }
    
    $params[] = $accountId;
    $query = "UPDATE accounts SET " . implode(', ', $updates) . " WHERE id = ?";
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    
    echo json_encode(['success' => true, 'message' => 'Account updated']);
    exit;
}

// Delete account (Seller only)
if ($method === 'DELETE' && isset($_GET['id'])) {
    $user = requireSeller();
    $accountId = $_GET['id'];
    
    // Verify ownership
    $stmt = $db->prepare("SELECT seller_id FROM accounts WHERE id = ?");
    $stmt->execute([$accountId]);
    $account = $stmt->fetch();
    
    if (!$account || $account['seller_id'] != $user['id']) {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'Not authorized']);
        exit;
    }
    
    $stmt = $db->prepare("DELETE FROM accounts WHERE id = ?");
    $stmt->execute([$accountId]);
    
    echo json_encode(['success' => true, 'message' => 'Account deleted']);
    exit;
}

http_response_code(405);
echo json_encode(['success' => false, 'message' => 'Method not allowed']);

