<?php
/**
 * Authentication API
 * Handles login, register, Google OAuth
 */

header('Content-Type: application/json');
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../jwt.php';
require_once __DIR__ . '/../encryption.php';
require_once __DIR__ . '/rate_limit.php';

// CORS Headers
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

$method = $_SERVER['REQUEST_METHOD'];
$db = getDB();

// Helper function to send JSON response
function sendResponse($success, $message, $data = null, $code = 200) {
    http_response_code($code);
    echo json_encode([
        'success' => $success,
        'message' => $message,
        'data' => $data
    ]);
    exit;
}

// Register
if ($method === 'POST' && isset($_GET['action']) && $_GET['action'] === 'register') {
    // Rate limit for registration
    $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    checkRateLimit('register_' . $ip, 5, 60); // 5 registrations per minute per IP
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    $name = trim($input['name'] ?? '');
    $email = trim($input['email'] ?? '');
    $password = $input['password'] ?? '';
    $role = $input['role'] ?? 'buyer';

    if (empty($name) || empty($email) || empty($password)) {
        sendResponse(false, 'All fields are required', null, 400);
    }

    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        sendResponse(false, 'Invalid email format', null, 400);
    }

    if (strlen($password) < 6) {
        sendResponse(false, 'Password must be at least 6 characters', null, 400);
    }

    // Check if email exists
    $stmt = $db->prepare("SELECT id FROM users WHERE email = ?");
    $stmt->execute([$email]);
    if ($stmt->fetch()) {
        sendResponse(false, 'Email already registered', null, 400);
    }

    // Create user
    $hashedPassword = password_hash($password, PASSWORD_BCRYPT);
    $stmt = $db->prepare("INSERT INTO users (name, email, password, role) VALUES (?, ?, ?, ?)");
    
    if ($stmt->execute([$name, $email, $hashedPassword, $role])) {
        $userId = $db->lastInsertId();
        
        // Generate JWT
        $token = JWT::generate(['user_id' => $userId, 'email' => $email, 'role' => $role]);
        
        // Get user data
        $stmt = $db->prepare("SELECT id, name, email, role, avatar, wallet_balance, is_verified FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
        
        sendResponse(true, 'Registration successful', [
            'token' => $token,
            'user' => $user
        ]);
    } else {
        sendResponse(false, 'Registration failed', null, 500);
    }
}

// Login
if ($method === 'POST' && isset($_GET['action']) && $_GET['action'] === 'login') {
    // Rate limit for login attempts
    $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
    checkRateLimit('login_' . $ip, 5, 60); // 5 login attempts per minute per IP
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    $email = trim($input['email'] ?? '');
    $password = $input['password'] ?? '';

    if (empty($email) || empty($password)) {
        sendResponse(false, 'Email and password are required', null, 400);
    }

    $stmt = $db->prepare("SELECT id, name, email, password, role, avatar, wallet_balance, is_verified, is_blocked FROM users WHERE email = ?");
    $stmt->execute([$email]);
    $user = $stmt->fetch();

    if (!$user) {
        sendResponse(false, 'Invalid credentials', null, 401);
    }

    if ($user['is_blocked']) {
        sendResponse(false, 'Your account has been blocked', null, 403);
    }

    if (!password_verify($password, $user['password'])) {
        sendResponse(false, 'Invalid credentials', null, 401);
    }

    // Generate JWT
    $token = JWT::generate([
        'user_id' => $user['id'],
        'email' => $user['email'],
        'role' => $user['role']
    ]);

    unset($user['password']);
    sendResponse(true, 'Login successful', [
        'token' => $token,
        'user' => $user
    ]);
}

// Google OAuth Login
if ($method === 'POST' && isset($_GET['action']) && $_GET['action'] === 'google') {
    $input = json_decode(file_get_contents('php://input'), true);
    $googleId = $input['google_id'] ?? '';
    $email = $input['email'] ?? '';
    $name = $input['name'] ?? '';
    $avatar = $input['avatar'] ?? '';

    if (empty($googleId) || empty($email)) {
        sendResponse(false, 'Google ID and email are required', null, 400);
    }

    // Check if user exists
    $stmt = $db->prepare("SELECT id, name, email, role, avatar, wallet_balance, is_verified, is_blocked FROM users WHERE google_id = ? OR email = ?");
    $stmt->execute([$googleId, $email]);
    $user = $stmt->fetch();

    if ($user) {
        if ($user['is_blocked']) {
            sendResponse(false, 'Your account has been blocked', null, 403);
        }
        
        // Update Google ID if not set
        if (empty($user['google_id'])) {
            $stmt = $db->prepare("UPDATE users SET google_id = ? WHERE id = ?");
            $stmt->execute([$googleId, $user['id']]);
        }
    } else {
        // Create new user
        $stmt = $db->prepare("INSERT INTO users (name, email, google_id, avatar, role, is_verified) VALUES (?, ?, ?, ?, 'buyer', 1)");
        $stmt->execute([$name, $email, $googleId, $avatar]);
        $userId = $db->lastInsertId();
        
        $stmt = $db->prepare("SELECT id, name, email, role, avatar, wallet_balance, is_verified FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch();
    }

    // Generate JWT
    $token = JWT::generate([
        'user_id' => $user['id'],
        'email' => $user['email'],
        'role' => $user['role']
    ]);

    sendResponse(true, 'Login successful', [
        'token' => $token,
        'user' => $user
    ]);
}

// Verify Token
if ($method === 'GET' && isset($_GET['action']) && $_GET['action'] === 'verify') {
    $token = JWT::getTokenFromHeader();
    
    if (!$token) {
        sendResponse(false, 'Token not provided', null, 401);
    }

    $payload = JWT::verify($token);
    if (!$payload) {
        sendResponse(false, 'Invalid or expired token', null, 401);
    }

    // Get user data
    $stmt = $db->prepare("SELECT id, name, email, role, avatar, wallet_balance, is_verified, is_blocked FROM users WHERE id = ?");
    $stmt->execute([$payload['user_id']]);
    $user = $stmt->fetch();

    if (!$user || $user['is_blocked']) {
        sendResponse(false, 'User not found or blocked', null, 401);
    }

    sendResponse(true, 'Token valid', ['user' => $user]);
}

sendResponse(false, 'Invalid request', null, 400);

