<?php
/**
 * Manual Payment Gateways Management API
 * Admin only - for managing manual payment gateways
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/middleware.php';
require_once __DIR__ . '/../database.php';

$db = getDB();
$method = $_SERVER['REQUEST_METHOD'];

// List gateways (public - for deposit selection)
if ($method === 'GET' && !isset($_GET['admin'])) {
    $stmt = $db->query("SELECT id, name, logo, payment_type, min_amount, max_amount, instructions FROM manual_gateways WHERE is_active = 1 ORDER BY sort_order");
    $gateways = $stmt->fetchAll();
    
    echo json_encode(['success' => true, 'data' => $gateways]);
    exit;
}

// Admin endpoints
$admin = requireAdmin();

// List all gateways (admin)
if ($method === 'GET' && isset($_GET['admin'])) {
    $stmt = $db->query("SELECT * FROM manual_gateways ORDER BY sort_order, name ASC");
    $gateways = $stmt->fetchAll();
    
    echo json_encode(['success' => true, 'data' => $gateways]);
    exit;
}

// Create gateway
if ($method === 'POST') {
    $input = json_decode(file_get_contents('php://input'), true);
    
    $name = trim($input['name'] ?? '');
    $logo = $input['logo'] ?? null;
    $paymentType = $input['payment_type'] ?? 'personal';
    $paymentNumber = trim($input['payment_number'] ?? '');
    $accountName = trim($input['account_name'] ?? '');
    $instructions = $input['instructions'] ?? '';
    $minAmount = floatval($input['min_amount'] ?? 0);
    $maxAmount = floatval($input['max_amount'] ?? 10000);
    $isActive = isset($input['is_active']) ? ($input['is_active'] ? 1 : 0) : 1;
    $sortOrder = intval($input['sort_order'] ?? 0);
    
    if (empty($name) || empty($paymentNumber)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Name and payment number are required']);
        exit;
    }
    
    $stmt = $db->prepare("
        INSERT INTO manual_gateways (name, logo, payment_type, payment_number, account_name, instructions, min_amount, max_amount, is_active, sort_order)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?)
    ");
    $stmt->execute([$name, $logo, $paymentType, $paymentNumber, $accountName, $instructions, $minAmount, $maxAmount, $isActive, $sortOrder]);
    
    $gatewayId = $db->lastInsertId();
    echo json_encode(['success' => true, 'message' => 'Gateway created', 'data' => ['id' => $gatewayId]]);
    exit;
}

// Update gateway
if ($method === 'PUT' && isset($_GET['id'])) {
    $id = intval($_GET['id']);
    $input = json_decode(file_get_contents('php://input'), true);
    
    $updates = [];
    $params = [];
    
    if (isset($input['name'])) {
        $updates[] = "name = ?";
        $params[] = trim($input['name']);
    }
    if (isset($input['logo'])) {
        $updates[] = "logo = ?";
        $params[] = $input['logo'];
    }
    if (isset($input['payment_type'])) {
        $updates[] = "payment_type = ?";
        $params[] = $input['payment_type'];
    }
    if (isset($input['payment_number'])) {
        $updates[] = "payment_number = ?";
        $params[] = trim($input['payment_number']);
    }
    if (isset($input['account_name'])) {
        $updates[] = "account_name = ?";
        $params[] = trim($input['account_name']);
    }
    if (isset($input['instructions'])) {
        $updates[] = "instructions = ?";
        $params[] = $input['instructions'];
    }
    if (isset($input['min_amount'])) {
        $updates[] = "min_amount = ?";
        $params[] = floatval($input['min_amount']);
    }
    if (isset($input['max_amount'])) {
        $updates[] = "max_amount = ?";
        $params[] = floatval($input['max_amount']);
    }
    if (isset($input['is_active'])) {
        $updates[] = "is_active = ?";
        $params[] = $input['is_active'] ? 1 : 0;
    }
    if (isset($input['sort_order'])) {
        $updates[] = "sort_order = ?";
        $params[] = intval($input['sort_order']);
    }
    
    if (empty($updates)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'No fields to update']);
        exit;
    }
    
    $params[] = $id;
    $query = "UPDATE manual_gateways SET " . implode(', ', $updates) . " WHERE id = ?";
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    
    echo json_encode(['success' => true, 'message' => 'Gateway updated']);
    exit;
}

// Delete gateway
if ($method === 'DELETE' && isset($_GET['id'])) {
    $id = intval($_GET['id']);
    
    $stmt = $db->prepare("DELETE FROM manual_gateways WHERE id = ?");
    $stmt->execute([$id]);
    
    echo json_encode(['success' => true, 'message' => 'Gateway deleted']);
    exit;
}

http_response_code(405);
echo json_encode(['success' => false, 'message' => 'Method not allowed']);

