<?php
/**
 * API Middleware
 * Authentication and authorization checks
 */

require_once __DIR__ . '/../jwt.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/rate_limit.php';

function requireAuth() {
    $token = JWT::getTokenFromHeader();
    
    if (!$token) {
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Authentication required']);
        exit;
    }

    $payload = JWT::verify($token);
    if (!$payload) {
        http_response_code(401);
        echo json_encode(['success' => false, 'message' => 'Invalid or expired token']);
        exit;
    }

    // Check if user exists and is not blocked
    $db = getDB();
    $stmt = $db->prepare("SELECT id, name, email, role, is_blocked FROM users WHERE id = ?");
    $stmt->execute([$payload['user_id']]);
    $user = $stmt->fetch();

    if (!$user || $user['is_blocked']) {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'User not found or blocked']);
        exit;
    }

    // Apply rate limiting
    applyRateLimit($user);

    return $user;
}

function requireRole($allowedRoles) {
    $user = requireAuth();
    
    if (!in_array($user['role'], $allowedRoles)) {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'Insufficient permissions']);
        exit;
    }

    return $user;
}

function requireAdmin() {
    return requireRole(['admin']);
}

function requireSeller() {
    return requireRole(['seller', 'admin']);
}

function requireBuyer() {
    return requireRole(['buyer', 'seller', 'admin']);
}

