<?php
/**
 * Orders API
 * Order creation, management, reviews
 */

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type, Authorization');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

require_once __DIR__ . '/middleware.php';
require_once __DIR__ . '/../database.php';
require_once __DIR__ . '/../encryption.php';

$db = getDB();
$method = $_SERVER['REQUEST_METHOD'];

// List orders
if ($method === 'GET' && !isset($_GET['id'])) {
    $user = requireAuth();
    $type = $_GET['type'] ?? 'all'; // all, buyer, seller
    $status = $_GET['status'] ?? null;
    $page = intval($_GET['page'] ?? 1);
    $limit = 20;
    $offset = ($page - 1) * $limit;
    
    $query = "";
    $params = [];
    
    if ($type === 'buyer' || ($user['role'] === 'buyer' && $type === 'all')) {
        $query = "
            SELECT o.*, a.title as account_title, a.screenshots,
                   u.name as seller_name, u.avatar as seller_avatar,
                   st.name as social_type_name
            FROM orders o
            JOIN accounts a ON o.account_id = a.id
            JOIN users u ON o.seller_id = u.id
            JOIN social_media_types st ON a.social_type_id = st.id
            WHERE o.buyer_id = ?
        ";
        $params[] = $user['id'];
    } elseif ($type === 'seller' || ($user['role'] === 'seller' && $type === 'all')) {
        $query = "
            SELECT o.*, a.title as account_title, a.screenshots,
                   u.name as buyer_name, u.avatar as buyer_avatar,
                   st.name as social_type_name
            FROM orders o
            JOIN accounts a ON o.account_id = a.id
            JOIN users u ON o.buyer_id = u.id
            JOIN social_media_types st ON a.social_type_id = st.id
            WHERE o.seller_id = ?
        ";
        $params[] = $user['id'];
    } else {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Invalid type']);
        exit;
    }
    
    if ($status) {
        $query .= " AND o.status = ?";
        $params[] = $status;
    }
    
    // Get total count
    $countQuery = "SELECT COUNT(*) as total FROM (" . $query . ") as subquery";
    $countStmt = $db->prepare($countQuery);
    $countStmt->execute($params);
    $total = (int)$countStmt->fetch()['total'];
    
    // Add pagination
    $query .= " ORDER BY o.created_at DESC LIMIT ? OFFSET ?";
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    $orders = $stmt->fetchAll();
    
    foreach ($orders as &$order) {
        $order['screenshots'] = json_decode($order['screenshots'] ?? '[]', true);
    }
    
    echo json_encode([
        'success' => true,
        'data' => $orders,
        'pagination' => [
            'page' => $page,
            'limit' => $limit,
            'total' => $total,
            'pages' => ceil($total / $limit)
        ]
    ]);
    exit;
}

// Get single order
if ($method === 'GET' && isset($_GET['id'])) {
    $user = requireAuth();
    $orderId = $_GET['id'];
    
    $stmt = $db->prepare("
        SELECT o.*, a.title, a.description, a.screenshots, a.credentials_data,
               u_buyer.name as buyer_name, u_seller.name as seller_name,
               st.name as social_type_name
        FROM orders o
        JOIN accounts a ON o.account_id = a.id
        JOIN users u_buyer ON o.buyer_id = u_buyer.id
        JOIN users u_seller ON o.seller_id = u_seller.id
        JOIN social_media_types st ON a.social_type_id = st.id
        WHERE o.id = ? AND (o.buyer_id = ? OR o.seller_id = ?)
    ");
    $stmt->execute([$orderId, $user['id'], $user['id']]);
    $order = $stmt->fetch();
    
    if (!$order) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Order not found']);
        exit;
    }
    
    $order['screenshots'] = json_decode($order['screenshots'] ?? '[]', true);
    
    // Decrypt credentials if order is delivered
    if ($order['status'] === 'delivered' && ($order['buyer_id'] == $user['id'] || $user['role'] === 'admin')) {
        $order['credentials'] = Encryption::decryptJson($order['credentials_data']);
    }
    
    unset($order['credentials_data']);
    
    echo json_encode(['success' => true, 'data' => $order]);
    exit;
}

// Create order (Buyer)
if ($method === 'POST') {
    $user = requireBuyer();
    $input = json_decode(file_get_contents('php://input'), true);
    
    $accountId = $input['account_id'] ?? null;
    $paymentMethod = $input['payment_method'] ?? 'wallet';
    
    if (empty($accountId)) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Account ID is required']);
        exit;
    }
    
    // Get account details
    $stmt = $db->prepare("
        SELECT a.*, st.commission_percent
        FROM accounts a
        JOIN social_media_types st ON a.social_type_id = st.id
        WHERE a.id = ? AND a.is_active = 1 AND a.stock > 0
    ");
    $stmt->execute([$accountId]);
    $account = $stmt->fetch();
    
    if (!$account) {
        http_response_code(404);
        echo json_encode(['success' => false, 'message' => 'Account not available']);
        exit;
    }
    
    if ($account['seller_id'] == $user['id']) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Cannot purchase your own account']);
        exit;
    }
    
    $amount = $account['price'];
    $commission = ($amount * $account['commission_percent']) / 100;
    $sellerEarnings = $amount - $commission;
    
    // Check wallet balance if paying with wallet
    if ($paymentMethod === 'wallet') {
        $stmt = $db->prepare("SELECT wallet_balance FROM users WHERE id = ?");
        $stmt->execute([$user['id']]);
        $buyer = $stmt->fetch();
        
        if ($buyer['wallet_balance'] < $amount) {
            http_response_code(400);
            echo json_encode(['success' => false, 'message' => 'Insufficient wallet balance']);
            exit;
        }
        
        // Deduct from wallet
        $newBalance = $buyer['wallet_balance'] - $amount;
        $stmt = $db->prepare("UPDATE users SET wallet_balance = ? WHERE id = ?");
        $stmt->execute([$newBalance, $user['id']]);
        
        // Record transaction
        $stmt = $db->prepare("
            INSERT INTO transactions (user_id, type, amount, balance_before, balance_after, description)
            VALUES (?, 'order_payment', ?, ?, ?, ?)
        ");
        $stmt->execute([$user['id'], -$amount, $buyer['wallet_balance'], $newBalance, "Order payment for account #{$accountId}"]);
        
        $paymentStatus = 'completed';
    } else {
        $paymentStatus = 'pending';
    }
    
    // Create order
    $orderNumber = 'ORD' . time() . rand(1000, 9999);
    $stmt = $db->prepare("
        INSERT INTO orders (order_number, buyer_id, seller_id, account_id, amount, commission, seller_earnings, payment_method, payment_status, status)
        VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending')
    ");
    $stmt->execute([$orderNumber, $user['id'], $account['seller_id'], $accountId, $amount, $commission, $sellerEarnings, $paymentMethod, $paymentStatus]);
    
    $orderId = $db->lastInsertId();
    
    // Update account stock
    $stmt = $db->prepare("UPDATE accounts SET stock = stock - 1, sold_count = sold_count + 1 WHERE id = ?");
    $stmt->execute([$accountId]);
    
    // If payment completed, credit seller
    if ($paymentStatus === 'completed') {
        $stmt = $db->prepare("SELECT wallet_balance FROM users WHERE id = ?");
        $stmt->execute([$account['seller_id']]);
        $seller = $stmt->fetch();
        
        $newSellerBalance = $seller['wallet_balance'] + $sellerEarnings;
        $stmt = $db->prepare("UPDATE users SET wallet_balance = ? WHERE id = ?");
        $stmt->execute([$newSellerBalance, $account['seller_id']]);
        
        // Record transaction
        $stmt = $db->prepare("
            INSERT INTO transactions (user_id, type, amount, balance_before, balance_after, reference_id, description)
            VALUES (?, 'order_earning', ?, ?, ?, ?, ?)
        ");
        $stmt->execute([$account['seller_id'], $sellerEarnings, $seller['wallet_balance'], $newSellerBalance, $orderId, "Order earning from order #{$orderNumber}"]);
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Order created successfully',
        'data' => [
            'order_id' => $orderId,
            'order_number' => $orderNumber,
            'payment_status' => $paymentStatus,
            'payment_method' => $paymentMethod
        ]
    ]);
    exit;
}

// Mark order as delivered (Seller)
if ($method === 'PUT' && isset($_GET['id']) && isset($_GET['action']) && $_GET['action'] === 'deliver') {
    $user = requireSeller();
    $orderId = $_GET['id'];
    
    // Verify seller owns this order
    $stmt = $db->prepare("SELECT seller_id, payment_status FROM orders WHERE id = ?");
    $stmt->execute([$orderId]);
    $order = $stmt->fetch();
    
    if (!$order || $order['seller_id'] != $user['id']) {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'Not authorized']);
        exit;
    }
    
    if ($order['payment_status'] !== 'completed') {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Payment not completed']);
        exit;
    }
    
    $stmt = $db->prepare("UPDATE orders SET status = 'delivered' WHERE id = ?");
    $stmt->execute([$orderId]);
    
    // Credit seller if not already done
    $stmt = $db->prepare("SELECT seller_id, seller_earnings FROM orders WHERE id = ?");
    $stmt->execute([$orderId]);
    $order = $stmt->fetch();
    
    $stmt = $db->prepare("SELECT wallet_balance FROM users WHERE id = ?");
    $stmt->execute([$order['seller_id']]);
    $seller = $stmt->fetch();
    
    $newBalance = $seller['wallet_balance'] + $order['seller_earnings'];
    $stmt = $db->prepare("UPDATE users SET wallet_balance = ? WHERE id = ?");
    $stmt->execute([$newBalance, $order['seller_id']]);
    
    // Record transaction
    $stmt = $db->prepare("
        INSERT INTO transactions (user_id, type, amount, balance_before, balance_after, reference_id, description)
        VALUES (?, 'order_earning', ?, ?, ?, ?, ?)
    ");
    $stmt->execute([$order['seller_id'], $order['seller_earnings'], $seller['wallet_balance'], $newBalance, $orderId, "Order earning"]);
    
    echo json_encode(['success' => true, 'message' => 'Order marked as delivered']);
    exit;
}

// Add review (Buyer)
if ($method === 'POST' && isset($_GET['action']) && $_GET['action'] === 'review') {
    $user = requireBuyer();
    $input = json_decode(file_get_contents('php://input'), true);
    
    $orderId = $input['order_id'] ?? null;
    $rating = $input['rating'] ?? null;
    $comment = $input['comment'] ?? '';
    
    if (empty($orderId) || empty($rating) || $rating < 1 || $rating > 5) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Invalid review data']);
        exit;
    }
    
    // Verify order belongs to buyer and is delivered
    $stmt = $db->prepare("SELECT buyer_id, seller_id, status FROM orders WHERE id = ?");
    $stmt->execute([$orderId]);
    $order = $stmt->fetch();
    
    if (!$order || $order['buyer_id'] != $user['id'] || $order['status'] !== 'delivered') {
        http_response_code(403);
        echo json_encode(['success' => false, 'message' => 'Cannot review this order']);
        exit;
    }
    
    // Check if review already exists
    $stmt = $db->prepare("SELECT id FROM reviews WHERE order_id = ?");
    $stmt->execute([$orderId]);
    if ($stmt->fetch()) {
        http_response_code(400);
        echo json_encode(['success' => false, 'message' => 'Review already exists']);
        exit;
    }
    
    $stmt = $db->prepare("INSERT INTO reviews (order_id, buyer_id, seller_id, rating, comment) VALUES (?, ?, ?, ?, ?)");
    $stmt->execute([$orderId, $user['id'], $order['seller_id'], $rating, $comment]);
    
    echo json_encode(['success' => true, 'message' => 'Review added successfully']);
    exit;
}

http_response_code(405);
echo json_encode(['success' => false, 'message' => 'Method not allowed']);

